--[[
	SowingMachineRollerReady.lua
	
	Autor: 		Ifko[nator]
	Datum: 		15.09.2025
	Version: 	2.1
	
	Changelog:	v1.0 @24.12.2021 - initial implementation in FS 22
				------------------------------------------------------------------------------------------
				v1.1 @22.04.2022 - fix for patch 1.4 and higher
				------------------------------------------------------------------------------------------
				v1.2 @18.07.2022 - fix for sowing machines with fertilizer
				------------------------------------------------------------------------------------------
				v2.0 @29.06.2025 - convert to FS 25
				------------------------------------------------------------------------------------------
				v2.1 @15.09.2025 - fix for sowing machines with fertilizing function and precision farming
]]

SowingMachineRollerReady = {};
SowingMachineRollerReady.currentModName = "";
SowingMachineRollerReady.precisionFarmingModTable = {};

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "SOWING MACHINE ROLLER READY" or mod.title:upper() == "WALZFUNKTION FÜR SÄMASCHINEN" then		
		if g_modIsLoaded[mod.modName] then	
			SowingMachineRollerReady.currentModName = mod.modName;
			
			break;
		end;
	end;
end;

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "PRECISION FARMING" then		
		if g_modIsLoaded[mod.modName] then	
			SowingMachineRollerReady.precisionFarmingModTable = _G[mod.modName];
			
			break;
		end;
	end;
end;

function SowingMachineRollerReady.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).sowingMachineRollerReady#allowRollerSowingMachine", "Allow/Disallow roller function.", false);
end;

function SowingMachineRollerReady.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(SowingMachine, specializations);
end;

function SowingMachineRollerReady.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdateTick",
		"onReadStream",
		"onWriteStream",
		"onRegisterActionEvents"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, SowingMachineRollerReady);
	end;
end;

function SowingMachineRollerReady.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "setAllowRollerSowingMachine", SowingMachineRollerReady.setAllowRollerSowingMachine);
end;

function SowingMachineRollerReady:onLoad(savegame)
	local specSowingMachine = self.spec_sowingMachine;
	
	specSowingMachine.isRollerAllowed = self.xmlFile:getValue("vehicle.sowingMachine.fieldGroundType#value") == "SOWN";
	specSowingMachine.allowRollerSowingMachine = false;

	if not specSowingMachine.isRollerAllowed then
		return;
	end;

	if savegame ~= nil and not savegame.resetVehicles then
		specSowingMachine.allowRollerSowingMachine = savegame.xmlFile:getValue(savegame.key .. ".sowingMachineRollerReady#allowRollerSowingMachine", specSowingMachine.allowRollerSowingMachine);
	end;

	self:setAllowRollerSowingMachine(specSowingMachine.allowRollerSowingMachine, false);

	specSowingMachine.l10nTexts = {};

	local l10nTexts = {
		"action_activateRoller",
		"action_deactivateRoller"
	};

	for _, l10nText in pairs(l10nTexts) do
		specSowingMachine.l10nTexts[l10nText] = g_i18n:getText(l10nText, SowingMachineRollerReady.currentModName);
	end;
end;

function SowingMachineRollerReady:onReadStream(streamId, connection)
	self.spec_sowingMachine.allowRollerSowingMachine = streamReadBool(streamId);
end;

function SowingMachineRollerReady:onWriteStream(streamId, connection)
	streamWriteBool(streamId, self.spec_sowingMachine.allowRollerSowingMachine);
end;

function SowingMachineRollerReady:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
		local specSowingMachine = self.spec_sowingMachine;

		if isActiveForInputIgnoreSelection and specSowingMachine.isRollerAllowed then
			local _, actionEventId = self:addActionEvent(specSowingMachine.actionEvents, InputAction.SET_ALLOW_ROLLER_BUTTON, self, SowingMachineRollerReady.actionEventSetAllowRollerSowingMachine, false, true, false, true, nil);
			
			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH);
			g_inputBinding:setActionEventTextVisibility(actionEventId, true);
			g_inputBinding:setActionEventActive(actionEventId, true);
			g_inputBinding:setActionEventText(actionEventId, specSowingMachine.l10nTexts.action_activateRoller);
		end;
	end;
end;

function SowingMachineRollerReady:onUpdateTick(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local specSowingMachine = self.spec_sowingMachine;

	if self:getFillUnitFillLevel(specSowingMachine.fillUnitIndex) <= 0 and self:getFillUnitCapacity(specSowingMachine.fillUnitIndex) ~= 0 then
		if self:getIsTurnedOn() then
			g_currentMission:showBlinkingWarning(g_i18n:getText("warning_pleaseFillFirstSowingMachine", SowingMachineRollerReady.currentModName), 5000);

			self:setIsTurnedOn(false);
		end;
	end;

	if not specSowingMachine.isRollerAllowed then
		return;
	end;

	local actionEvent = specSowingMachine.actionEvents[InputAction.SET_ALLOW_ROLLER_BUTTON];

	if actionEvent ~= nil then
		local actionEventText = specSowingMachine.l10nTexts.action_activateRoller;

		if specSowingMachine.allowRollerSowingMachine then
			actionEventText = specSowingMachine.l10nTexts.action_deactivateRoller;
		end;

		g_inputBinding:setActionEventText(actionEvent.actionEventId, actionEventText);
	end;
end;

function SowingMachineRollerReady.actionEventSetAllowRollerSowingMachine(self, actionName, inputValue, callbackState, isAnalog)
	self:setAllowRollerSowingMachine(not self.spec_sowingMachine.allowRollerSowingMachine, false);
end;

function SowingMachineRollerReady:setAllowRollerSowingMachine(allowRollerSowingMachine, noEventSend)
	self.spec_sowingMachine.allowRollerSowingMachine = allowRollerSowingMachine;
	
	SowingMachineRollerReadyEvent.sendEvent(self, allowRollerSowingMachine, noEventSend);
end;

function SowingMachineRollerReady:saveToXMLFile(xmlFile, key)
	local specSowingMachine = self.spec_sowingMachine;
	
	if specSowingMachine.allowRollerSowingMachine then	
		xmlFile:setValue(key .. "#allowRollerSowingMachine", specSowingMachine.allowRollerSowingMachine);
	end;
end;

function SowingMachineRollerReady:processSowingMachineArea(superFunc, workArea, dt)
	local changedArea, totalArea = superFunc(self, workArea, dt);

	local specSowingMachine = self.spec_sowingMachine;
	
	if specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		changedArea, _ = changedArea + FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
		
		specSowingMachine.workAreaParameters.lastChangedArea = specSowingMachine.workAreaParameters.lastChangedArea + changedArea;
		specSowingMachine.workAreaParameters.lastStatsArea = specSowingMachine.workAreaParameters.lastStatsArea + changedArea;
	end;

	return changedArea, totalArea;
end;

SowingMachine.processSowingMachineArea = Utils.overwrittenFunction(SowingMachine.processSowingMachineArea, SowingMachineRollerReady.processSowingMachineArea);

function SowingMachineRollerReady:processFertilizingSowingMachineArea(superFunc, superFuncFertilizingSowingMachine, workArea, dt)
	local changedArea, totalArea = superFunc(self, workArea, dt);

	local specSowingMachine = self.spec_sowingMachine;

	if specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		changedArea, _ = changedArea + FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
		
		specSowingMachine.workAreaParameters.lastChangedArea = specSowingMachine.workAreaParameters.lastChangedArea + changedArea;
		specSowingMachine.workAreaParameters.lastStatsArea = specSowingMachine.workAreaParameters.lastStatsArea + changedArea;
	end;
	
	
	return changedArea, totalArea;
end;

function SowingMachineRollerReady:preProcessExtUnderRootFertilizerArea(superFunc, workArea, dt)
    superFunc(self, workArea, dt);

	local specSowingMachine = self.spec_sowingMachine;

	if specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
	end;
end

if SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer == nil then
	FertilizingSowingMachine.processSowingMachineArea = Utils.overwrittenFunction(FertilizingSowingMachine.processSowingMachineArea, SowingMachineRollerReady.processFertilizingSowingMachineArea);
else
	SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer.preProcessExtUnderRootFertilizerArea = Utils.overwrittenFunction(SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer.preProcessExtUnderRootFertilizerArea, SowingMachineRollerReady.preProcessExtUnderRootFertilizerArea);
end;